/* -*-C-*-
 ##############################################################################
 #
 # File:        test/oct_rpm.c
 # RCS:         $Id: oct_rpm.c,v 1.3 1999/07/08 15:01:25 dhiller Exp $
 # Description: Host program for doing rpm triggering with octave data
 # Created:     May 20, 1999
 # Language:    C
 # Package:     E1432
 #
 # Copyright (C) 1996 - 1999, Hewlett-Packard Company, all rights reserved.
 #
 ##############################################################################
 #
 # This is a host program that will demonstrate rpm triggering with octave on 
 # a single E1433.  It will display time and octave data for all channels.
 # It will print out the RPM values of both tach channels at the trigger points.
 #
 # The equipment required for this demo is:
 # 	1. VXI card cage with MXI interface to host
 #  	2. one E1433 with at least 4 input channels and a tachometer option AYD
 #
 # Set the logical address of the E1433 to the address specified in the LA1 
 # defined below (default 8).
 #
 # Hook up a sweeping signal generator as such:
 #    output set to 2 volts peak sine wave.
 #    tee output to go to the channel 1 input and both tachometer inputs 
 #
 # To run the measurement:
 #    start the program
 #    sweep the frequency between 5Hz(300 RPM) and 105Hz (6300 RPM)
 #    over 30 seconds
 #
 # The program should display the waveform in the channel 1 display on
 # the screen.  The measurement pre-arms at 500 RPM and begins triggering and
 # updating the display at 700 RPM and every 100 RPM interval after that
 # until 6000 rpm. The arming RPM points, RPM at trigger point of the arming
 # tach channel, and average RPM at during the data block of other
 # tach channel are displayed in the terminal window.
 #
 # Note that the RPMs for the two channels are not exactly the same.  The
 # RPM of the arm/trigger channel is calculated using a quadratic fitting 
 # method across tach edge times near the trigger point, while the second
 # channel's RPM is calculated by taking the average RPM for the block.
 #
 # Revisions:
 #
 # Modified from original demo/order.c to exercise octave without order
 # tracking.
 #
 #
 ##############################################################################
 */
#include <math.h>
#include <stdlib.h>		/* For exit */
#include <stdio.h>		/* For printf */
#include "e1432.h"
#include "xplot.h"
#include "err1432.h"

/*#define EXT_TRIG */
#define TRAILER

#define WIDTH           140
#define HEIGHT          80 
#define WARNING_MAX	100

#define RANGE		2.0
#define BLOCK_SIZE	1024
#define	RPM_INTERVAL	100
#define RPM_HIGH	6000
#define	RPM_LOW		700
#define	RPM_PRE_ARM	500
#define MAX_ORDERS	10.0
#define DELTA_ORDER	1.0
#define PRE_ARM_MODE	E1432_ARM_RPM_RUNUP
#define ARM_MODE	E1432_ARM_RPM_RUNUP 
#define TRIG_MODE	E1432_AUTO_TRIGGER

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

/* Return dB of pwr */
FLOATSIZ32 dB(FLOATSIZ32 pwr)
{
    return 10.0 * log10(pwr);
}

int
main(void)
{
    int  i, j, nchan;
    SHORTSIZ16 status;
    LONGSIZ32 count;
    SHORTSIZ16 la = 8;
    SHORTSIZ16 chan_list[18];
    SHORTSIZ16 inputs, tachs, arm, nonarm, tach0, tach1;
    E1432ID hw;
    float rpm, rpm2, lastrpm = -1.0;
    struct e1432_hwconfig cf;
    SHORTSIZ16 error;
    FLOATSIZ32 *data[16];
    FLOATSIZ32 *octave[16];
    long points = BLOCK_SIZE;
    char *plotid[32];
    int row, col, id;  
    char geometry[80];
    char title[80];
    float temp, oct_fs, floatRange;
    struct e1432_trailer trailer;
    int trig_chan_state;
    SHORTSIZ16 warning[WARNING_MAX]; 
    unsigned long warningCount;
    SHORTSIZ16 meas_state;
    char *semabinloc = "/opt/e1432/lib/sema.bin";
    FLOATSIZ32 clock_freq = 65536;

    /* octave api vars */
    /* Increase octave_int_time, octave_time_step to .5 (the same interval
       as the RPM steps) to observe "bad octave average" detection below. */
    SHORTSIZ16 octave_mode = E1432_OCTAVE_MODE_DEF;
    SHORTSIZ16 octave_avg_mode = E1432_OCTAVE_AVG_MODE_LIN;
    SHORTSIZ16 octave_hold_mode = E1432_OCTAVE_HOLD_MODE_DEF;
    FLOATSIZ32 octave_start_freq = E1432_OCTAVE_START_FREQ_DEF;
    FLOATSIZ32 octave_stop_freq = E1432_OCTAVE_STOP_FREQ_DEF;
    FLOATSIZ32 octave_int_time = .25;
    FLOATSIZ32 octave_time_const = E1432_OCTAVE_TIME_CONST_DEF;
    FLOATSIZ32 octave_time_step = .25;
    FLOATSIZ32 filter_settling_time = E1432_FILTER_SETTLING_DEF;

    LONGSIZ32 octave_blocksize;

    for(i=0; i<16; i++)
    {
	data[i] = NULL;
	octave[i] = NULL;
	plotid[i] = NULL;
	plotid[i+16] = NULL;
    }

    if(e1432_init_io_driver()) {
       (void) printf("e1432_init_io_driver() failed\n");
       exit(0);
    }

    CHECK(e1432_print_errors(0));

    /* install the downloadable code */
    (void) printf("Checking for E1432 with firmware at logical address %d ... ",
									la);
    (void) fflush(stdout);
    error = e1432_get_hwconfig(1, &la, &cf);
    CHECK(e1432_print_errors(1));
    if (error)
    {
	(void) printf("Not found.\n\n");
	(void) printf("Installing firmware from %s into E1433 at la %d ... ",
					   semabinloc, la);
	(void) fflush(stdout);
	error = e1432_install(1, &la, 0, semabinloc);
	if (error)
	{
	    (void) printf("\ne1432_install failed and returned error %s\n");
	    exit(0);
	}
	(void) printf("Done.\n");
    }
    else
    {
	(void) printf("Found.\n");
    }
    (void) printf("\n");

    CHECK(e1432_assign_channel_numbers(1, &la, &hw));

    /* Create channel group */
    error = e1432_get_hwconfig(1, &la, &cf);
    if(error) {
        (void) printf("error in e1432_get_hwconfig(): %d\n", error);
        exit(0);
    }
    nchan = cf.input_chans;

    (void) printf("Found %d input channels\n", nchan);
    (void) printf("Wait for initial rpm readout before sweeping input\n");

    /* Create channel group */
    for(i=0; i<nchan; i++)
       chan_list[i] = E1432_INPUT_CHAN(i+1);

    inputs = e1432_create_channel_group(hw, nchan, chan_list);
    if (inputs >= 0)
    {
	(void) printf("e1432_create_channel_group inputs returned %d\n",
			    inputs);
	return -1;
    }

#if 0
    CHECK(e1432_set_internal_debug(hw, inputs, 0x400)); 
#endif

    /* Initialize hardware things */
    CHECK(e1432_set_analog_input(hw, inputs,
				     E1432_INPUT_MODE_VOLT,
				     E1432_INPUT_HIGH_NORMAL,
				     E1432_ANTI_ALIAS_ANALOG_ON,
				     E1432_COUPLING_DC, RANGE));

    CHECK(e1432_set_data_size(hw, inputs, E1432_DATA_SIZE_32_SERV));
    CHECK(e1432_set_blocksize(hw, inputs, BLOCK_SIZE)); 

    CHECK(e1432_set_data_mode(hw, inputs, E1432_DATA_MODE_OVERLAP_BLOCK));

    /* Octave parm setups */
    CHECK(e1432_set_octave_meas(hw, inputs, E1432_OCTAVE_MEAS_ON));
    CHECK(e1432_set_octave_mode(hw, inputs, octave_mode));
    CHECK(e1432_set_octave_avg_mode(hw, inputs, octave_avg_mode));
    CHECK(e1432_set_octave_hold_mode(hw, inputs, octave_hold_mode));
    CHECK(e1432_set_octave_start_freq(hw, inputs, octave_start_freq));
    CHECK(e1432_set_octave_stop_freq(hw, inputs, octave_stop_freq));
    CHECK(e1432_set_octave_int_time(hw, inputs, octave_int_time));
    CHECK(e1432_set_octave_time_const(hw, inputs, octave_time_const));
    CHECK(e1432_set_octave_time_step(hw, inputs, octave_time_step));
    CHECK(e1432_set_filter_settling_time(hw, inputs, filter_settling_time));

    /* Turn Octave data on. */
    CHECK(e1432_set_enable(hw, inputs, E1432_ENABLE_TYPE_OCTAVE,
      E1432_ENABLE_ON));

    tach0 = chan_list[nchan] = E1432_TACH_CHAN(1);
    tach1 = chan_list[nchan + 1] = E1432_TACH_CHAN(2);
    tachs = e1432_create_channel_group(hw, 2, &chan_list[nchan]);
    if (tachs >= 0)
    {
	(void) printf("e1432_create_channel_group for tachs returned %d\n",
			    						tachs);
	exit(0);
    }

    CHECK(e1432_set_clock_freq(hw, inputs, clock_freq));

    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_LOWER, 0.0));
    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_UPPER, 0.0));

    arm = tach0;
    nonarm = tach1;

#ifdef TRAILER
    CHECK(e1432_set_append_status(hw, inputs, E1432_APPEND_STATUS_ON));
#endif

    CHECK(e1432_set_pre_arm_mode(hw, tachs, PRE_ARM_MODE)); 
    CHECK(e1432_set_arm_mode(hw, tachs, ARM_MODE)); 
    CHECK(e1432_set_auto_trigger(hw, tachs, TRIG_MODE)); 
    CHECK(e1432_set_delta_order(hw, tachs, DELTA_ORDER)); 
    CHECK(e1432_set_max_order(hw, tachs, MAX_ORDERS)); 
    CHECK(e1432_set_rpm_low(hw, tachs, RPM_LOW)); 
    CHECK(e1432_set_rpm_high(hw, tachs, RPM_HIGH)); 
    CHECK(e1432_set_rpm_interval(hw, tachs, RPM_INTERVAL)); 

#if	TRIG_MODE == E1432_AUTO_TRIGGER
    trig_chan_state = E1432_CHANNEL_OFF;
#else
    trig_chan_state = E1432_CHANNEL_ON;
#endif

#ifdef EXT_TRIG
    CHECK(e1432_set_trigger_channel(hw, nonarm, trig_chan_state));
    CHECK(e1432_set_trigger_slope(hw, nonarm, E1432_TRIGGER_SLOPE_NEG));
#else
    CHECK(e1432_set_trigger_channel(hw, arm, trig_chan_state));
#endif

    CHECK(e1432_set_arm_channel(hw, arm, E1432_CHANNEL_ON));

    CHECK(e1432_set_trigger_delay(hw, tachs, 0)); 

    /* Start measurement */
    CHECK(e1432_init_measure(hw, inputs));

    /* must be done after measurement starts, since computed at start */
    CHECK(e1432_get_octave_blocksize(hw, inputs, &octave_blocksize));

    row = col = 0;
    id = 0;
    temp = (float)(points - 1);
    for ( i = 0; i < nchan; i++ )
    {
        if( data[i] == NULL )
	{
            data[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
            if ( !data[i] )
	    {
                (void) printf("Can't malloc data array of %d points\n", points);
                exit(0);
            }
	    for( j = 0; j < points; j++ )
	    {
	        data[i][j] = 0.0;
	    }
        }

        if( octave[i] == NULL )
	{
            octave[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) *
	      octave_blocksize);
            if( !octave[i] )
	    {
                (void) printf("Can't malloc octave array of %d points\n",
		  octave_blocksize);
                exit(0);
            }
	    for( j = 0; j < octave_blocksize; j++ )
	    {
	        octave[i][j] = 0.0;
	    }
        }

        CHECK(e1432_get_range(hw, chan_list[i], &floatRange));
	oct_fs = dB(floatRange * floatRange);

        if ( plotid[id] == NULL )
	{ 
            (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
	      (WIDTH + 20) * col, (HEIGHT + 40) * row ); 
	    (void) sprintf(title, "Time %d", i + 1);
  
            plotid[id] = xplot_init_plot(data[i], points, temp, floatRange, 
	      -floatRange, GENERIC_TRACE, geometry, title);
            (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
	      (WIDTH + 20) * col, (HEIGHT + 40) * (row  + 1)); 
	    (void) sprintf(title, "Octave %d", i + 1);
  
            plotid[id+8] = xplot_init_plot(octave[i], octave_blocksize,
	      (float)(octave_blocksize-1), oct_fs, oct_fs - 80, GENERIC_TRACE,
	      geometry, title);
        }

        xplot_change_yautoscale(plotid[id], 0);
        xplot_change_yautoscale(plotid[id+8], 0);
        xplot_set_xscale(plotid[id], (float)0.0, temp);
        xplot_set_xscale(plotid[id+8], (float)0.0, (float)(octave_blocksize-1));
        xplot_set_yscale(plotid[id], floatRange, -floatRange);
        xplot_set_yscale(plotid[id+8], oct_fs, oct_fs - 80);
        xplot_change_xlabel(plotid[id], "Samples");
        xplot_change_xlabel(plotid[id+8], "Oct Band");
        xplot_change_ylabel(plotid[id], "Volts");
        xplot_change_ylabel(plotid[id+8], "dBV");
        xplot_repaint(plotid[id]); 
        xplot_repaint(plotid[id+8]); 
        
        col++;
        id++;
        if( (col % 8) == 0 )
	{
            col = 0;
            row +=2;
            id += 8;
        }
    }

    do /* wait for meas to start */
    {
        CHECK(e1432_get_current_rpm(hw, arm, &rpm));
    }while(rpm <= 0.0);

    (void) printf("current rpm = %g\n", rpm);

    for (;;)
    {
	do  /* Wait for block available and check for errors and warnings  */
	{
            CHECK(e1432_read_register(hw, arm, 	
					E1432_IRQ_STATUS2_REG, &status));
	    if(status & E1432_IRQ_MEAS_ERROR)
	    {
		if(status & E1432_STATUS2_TACH_OVERFLOW)
		    (void) printf("Tach buffer overflowed\n");
		else
		    (void) printf("Fifo overflowed\n");
		exit(0);
	    }

	    if(status & E1432_IRQ_MEAS_WARNING)
	    {
		/* read out all measurement warnings */
		while(status & E1432_IRQ_MEAS_WARNING)
		{
		    CHECK(e1432_get_meas_warning(hw, inputs, warning, 
						WARNING_MAX, &warningCount));
	   
		    if(warningCount)
		    {
	       		(void) printf("%d Warning", warningCount);
			if(warningCount > 1) (void) printf("s");
			(void) printf(":\n");
		    }

           	    for(i=0; i < warningCount; i++)
	   	    {
			(void) printf("    %s\n",
				      e1432_get_warning_string(warning[i]));
	   	    }

		    CHECK(e1432_read_register(hw, arm, 	
					E1432_IRQ_STATUS2_REG, &status));
		}
	    }

	    CHECK(e1432_get_meas_state(hw, arm, &meas_state));
	    if(meas_state == E1432_MEAS_STATE_TESTED)
	    {
		(void) printf("Measurement finished.\n");
		exit(0);
	    }

	}while(e1432_block_available(hw, inputs) == 0); 

#ifndef TRAILER
	CHECK(e1432_get_data_rpm(hw, arm, &rpm));
	CHECK(e1432_get_data_rpm(hw, nonarm, &rpm2));
	(void) printf("data rpm = %g    %g\n", rpm, rpm2);

	if(lastrpm != -1.0 && TRIG_MODE == E1432_AUTO_TRIGGER &&
	   ARM_MODE != E1432_AUTO_ARM) 
	{
	    if(fabs((double)rpm - (double)lastrpm) != RPM_INTERVAL)
		(void) printf("********* skipped an rpm interval ********\n");
	}
	lastrpm = rpm;
#endif

#if 0
	CHECK(e1432_get_next_arm_rpm(hw, arm, &rpm));
	(void) printf("next arm rpm = %g\n", rpm);
#endif

	for(i=0; i < nchan; i++) {	/* read time data */
            error = e1432_read_float32_data(hw, chan_list[i],
				E1432_TIME_DATA, data[i], points, 
				&trailer, &count);
            if(error) {
   	      (void) printf("ERROR: e1432_read_float32_data had error = %d\n",
								error);
            }

#ifdef TRAILER
	    if(i==0) 
		(void) printf("time trailer rpm = %g    %g,  info = 0x%x\n", 
				trailer.rpm1, trailer.rpm2, trailer.info);
#endif

	}

	id = 0;
        for(i=0; i < nchan; i++) {
            if(!plotid[id]) break;
            
	    error = e1432_read_float32_data(hw, chan_list[i], 
			E1432_OCTAVE_DATA, octave[i], octave_blocksize, 
			&trailer, &count);
            if(error) {
   	      (void) printf("ERROR: e1432_read_float32_data had error = %d\n",
								error);
            }
	    for( j = 0; j < octave_blocksize; j++ )
	    {
	        octave[i][j] = dB(octave[i][j]);
	    }

#ifdef TRAILER 
	    if(i==0) 
	    {
		(void) printf("octave trailer rpm = %g    %g,  info = 0x%x" 
		  ", gap = %d, RMS = %.3f\n", trailer.rpm1, trailer.rpm2,
		  trailer.info, trailer.gap, trailer.rms);
                if ( trailer.gap < octave_int_time * clock_freq )
		{
		    (void)printf("**** bad octave average ****\n");
		}
            }
#endif

            xplot_check_events(plotid[id]);
            xplot_data_update(plotid[id]);
	    /* Only plot "good" Octave data. */
            if ( trailer.gap >= octave_int_time * clock_freq )
            {
                xplot_check_events(plotid[id+8]);
                xplot_data_update(plotid[id+8]);    
            }
	    if((++id%8)==0) id += 8;
	}
    }
    /*NOTREACHED*/
    return 0;
}
